/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.s3.Block;
import org.apache.hadoop.fs.s3.FileSystemStore;
import org.apache.hadoop.fs.s3.INode;
import org.apache.hadoop.fs.s3.Jets3tFileSystemStore;
import org.apache.hadoop.fs.s3.S3Exception;
import org.apache.hadoop.fs.s3.S3InputStream;
import org.apache.hadoop.fs.s3.S3OutputStream;
import org.apache.hadoop.io.retry.RetryPolicies;
import org.apache.hadoop.io.retry.RetryPolicy;
import org.apache.hadoop.io.retry.RetryProxy;
import org.apache.hadoop.util.Progressable;

public class S3FileSystem
extends FileSystem {
    private static final long DEFAULT_BLOCK_SIZE = 0x4000000L;
    private URI uri;
    private FileSystemStore store;
    private FileSystem localFs;
    private Path workingDir = new Path("/user", System.getProperty("user.name"));

    public S3FileSystem() {
    }

    public S3FileSystem(FileSystemStore store) {
        this.store = store;
    }

    public URI getUri() {
        return this.uri;
    }

    public void initialize(URI uri, Configuration conf) throws IOException {
        if (this.store == null) {
            this.store = S3FileSystem.createDefaultStore(conf);
        }
        this.store.initialize(uri, conf);
        this.setConf(conf);
        this.uri = URI.create(uri.getScheme() + "://" + uri.getAuthority());
        this.localFs = S3FileSystem.get(URI.create("file:///"), conf);
    }

    private static FileSystemStore createDefaultStore(Configuration conf) {
        Jets3tFileSystemStore store = new Jets3tFileSystemStore();
        RetryPolicy basePolicy = RetryPolicies.retryUpToMaximumCountWithFixedSleep(conf.getInt("fs.s3.maxRetries", 4), conf.getLong("fs.s3.sleepTimeSeconds", 10L), TimeUnit.SECONDS);
        HashMap<Class<? extends Exception>, RetryPolicy> exceptionToPolicyMap = new HashMap<Class<? extends Exception>, RetryPolicy>();
        exceptionToPolicyMap.put(IOException.class, basePolicy);
        exceptionToPolicyMap.put(S3Exception.class, basePolicy);
        RetryPolicy methodPolicy = RetryPolicies.retryByException(RetryPolicies.TRY_ONCE_THEN_FAIL, exceptionToPolicyMap);
        HashMap<String, RetryPolicy> methodNameToPolicyMap = new HashMap<String, RetryPolicy>();
        methodNameToPolicyMap.put("storeBlock", methodPolicy);
        methodNameToPolicyMap.put("retrieveBlock", methodPolicy);
        return (FileSystemStore)RetryProxy.create(FileSystemStore.class, (Object)store, methodNameToPolicyMap);
    }

    public String getName() {
        return this.getUri().toString();
    }

    public Path getWorkingDirectory() {
        return this.workingDir;
    }

    public void setWorkingDirectory(Path dir) {
        this.workingDir = this.makeAbsolute(dir);
    }

    private Path makeAbsolute(Path path) {
        if (path.isAbsolute()) {
            return path;
        }
        return new Path(this.workingDir, path);
    }

    public boolean mkdirs(Path path, FsPermission permission) throws IOException {
        Path absolutePath = this.makeAbsolute(path);
        INode inode = this.store.retrieveINode(absolutePath);
        if (inode == null) {
            this.store.storeINode(absolutePath, INode.DIRECTORY_INODE);
        } else if (inode.isFile()) {
            throw new IOException(String.format("Can't make directory for path %s since it is a file.", absolutePath));
        }
        Path parent = absolutePath.getParent();
        return parent == null || this.mkdirs(parent);
    }

    public boolean isFile(Path path) throws IOException {
        INode inode = this.store.retrieveINode(this.makeAbsolute(path));
        if (inode == null) {
            return false;
        }
        return inode.isFile();
    }

    private INode checkFile(Path path) throws IOException {
        INode inode = this.store.retrieveINode(this.makeAbsolute(path));
        if (inode == null) {
            throw new IOException("No such file.");
        }
        if (inode.isDirectory()) {
            throw new IOException("Path " + path + " is a directory.");
        }
        return inode;
    }

    public FileStatus[] listStatus(Path f) throws IOException {
        Path absolutePath = this.makeAbsolute(f);
        INode inode = this.store.retrieveINode(absolutePath);
        if (inode == null) {
            return null;
        }
        if (inode.isFile()) {
            return new FileStatus[]{new S3FileStatus(f, inode)};
        }
        ArrayList<FileStatus> ret = new ArrayList<FileStatus>();
        for (Path p : this.store.listSubPaths(absolutePath)) {
            ret.add(this.getFileStatus(p.makeQualified(this)));
        }
        return ret.toArray(new FileStatus[0]);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public FSDataOutputStream create(Path file, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        INode inode = this.store.retrieveINode(this.makeAbsolute(file));
        if (inode != null) {
            if (!overwrite) throw new IOException("File already exists: " + file);
            this.delete(file);
            return new FSDataOutputStream(new S3OutputStream(this.getConf(), this.store, this.makeAbsolute(file), blockSize, progress, bufferSize), this.statistics);
        } else {
            Path parent = file.getParent();
            if (parent == null || this.mkdirs(parent)) return new FSDataOutputStream(new S3OutputStream(this.getConf(), this.store, this.makeAbsolute(file), blockSize, progress, bufferSize), this.statistics);
            throw new IOException("Mkdirs failed to create " + parent.toString());
        }
    }

    public FSDataInputStream open(Path path, int bufferSize) throws IOException {
        INode inode = this.checkFile(path);
        return new FSDataInputStream(new S3InputStream(this.getConf(), this.store, inode, this.statistics));
    }

    public boolean rename(Path src, Path dst) throws IOException {
        INode dstParentINode;
        Path absoluteSrc = this.makeAbsolute(src);
        INode srcINode = this.store.retrieveINode(absoluteSrc);
        if (srcINode == null) {
            return false;
        }
        Path absoluteDst = this.makeAbsolute(dst);
        INode dstINode = this.store.retrieveINode(absoluteDst);
        if (dstINode != null && dstINode.isDirectory()) {
            absoluteDst = new Path(absoluteDst, absoluteSrc.getName());
            dstINode = this.store.retrieveINode(absoluteDst);
        }
        if (dstINode != null) {
            return false;
        }
        Path dstParent = absoluteDst.getParent();
        if (dstParent != null && ((dstParentINode = this.store.retrieveINode(dstParent)) == null || dstParentINode.isFile())) {
            return false;
        }
        return this.renameRecursive(absoluteSrc, absoluteDst);
    }

    private boolean renameRecursive(Path src, Path dst) throws IOException {
        INode srcINode = this.store.retrieveINode(src);
        this.store.storeINode(dst, srcINode);
        this.store.deleteINode(src);
        if (srcINode.isDirectory()) {
            for (Path oldSrc : this.store.listDeepSubPaths(src)) {
                INode inode = this.store.retrieveINode(oldSrc);
                if (inode == null) {
                    return false;
                }
                Path newDst = new Path(oldSrc.toString().replaceFirst(src.toString(), dst.toString()));
                this.store.storeINode(newDst, inode);
                this.store.deleteINode(oldSrc);
            }
        }
        return true;
    }

    public boolean delete(Path path, boolean recursive) throws IOException {
        Path absolutePath = this.makeAbsolute(path);
        INode inode = this.store.retrieveINode(absolutePath);
        if (inode == null) {
            return false;
        }
        if (inode.isFile()) {
            this.store.deleteINode(absolutePath);
            for (Block block : inode.getBlocks()) {
                this.store.deleteBlock(block);
            }
        } else {
            FileStatus[] contents = this.listStatus(absolutePath);
            if (contents == null) {
                return false;
            }
            if (contents.length != 0 && !recursive) {
                throw new IOException("Directory " + path.toString() + " is not empty.");
            }
            for (FileStatus p : contents) {
                if (this.delete(p.getPath(), recursive)) continue;
                return false;
            }
            this.store.deleteINode(absolutePath);
        }
        return true;
    }

    @Deprecated
    public boolean delete(Path path) throws IOException {
        return this.delete(path, true);
    }

    public FileStatus getFileStatus(Path f) throws IOException {
        INode inode = this.store.retrieveINode(this.makeAbsolute(f));
        if (inode == null) {
            throw new FileNotFoundException(f + ": No such file or directory.");
        }
        return new S3FileStatus(f, inode);
    }

    void dump() throws IOException {
        this.store.dump();
    }

    void purge() throws IOException {
        this.store.purge();
    }

    private static class S3FileStatus
    extends FileStatus {
        S3FileStatus(Path f, INode inode) throws IOException {
            super(S3FileStatus.findLength(inode), inode.isDirectory(), 1, S3FileStatus.findBlocksize(inode), 0L, f);
        }

        private static long findLength(INode inode) {
            if (!inode.isDirectory()) {
                long length = 0L;
                for (Block block : inode.getBlocks()) {
                    length += block.getLength();
                }
                return length;
            }
            return 0L;
        }

        private static long findBlocksize(INode inode) {
            Block[] ret = inode.getBlocks();
            return ret == null ? 0L : ret[0].getLength();
        }
    }
}

